<?php

namespace GoDaddy\WordPress\MWC\Core\WooCommerce\Payments;

use Exception;
use GoDaddy\WordPress\MWC\Common\Configuration\Configuration;
use GoDaddy\WordPress\MWC\Common\Events\Events;
use GoDaddy\WordPress\MWC\Common\Helpers\ArrayHelper;
use GoDaddy\WordPress\MWC\Common\Register\Register;
use GoDaddy\WordPress\MWC\Common\Repositories\WooCommerceRepository;
use GoDaddy\WordPress\MWC\Common\Traits\Features\IsConditionalFeatureTrait;
use GoDaddy\WordPress\MWC\Core\Payments\API;
use GoDaddy\WordPress\MWC\Core\Payments\Poynt\Onboarding;
use GoDaddy\WordPress\MWC\Core\WooCommerce\Payments\Events\PaymentGatewayFirstActiveEvent;
use GoDaddy\WordPress\MWC\Core\WooCommerce\Payments\Events\Producers\OnboardingEventsProducer;
use GoDaddy\WordPress\MWC\Core\WooCommerce\Payments\Events\Producers\PaymentGatewayEventsProducer;
use GoDaddy\WordPress\MWC\Core\WooCommerce\Payments\Frontend\MyPaymentMethods;
use GoDaddy\WordPress\MWC\Core\WooCommerce\Payments\GoDaddyPayments\Frontend\Admin\PaymentMethodsListTable;

/**
 * Core payment gateways.
 *
 * Takes care of the necessary tasks for adding the core gateway(s) in a way that WooCommerce understands.
 *
 * @since 2.10.0
 */
class CorePaymentGateways
{
    use IsConditionalFeatureTrait;

    /** @var string[] classes to load as universal handlers */
    private $handlerClasses = [
        Captures::class,
        PaymentMethodsListTable::class,
        MyPaymentMethods::class,
        VirtualTerminal::class,
    ];

    /** @var string[] payments gateways to load */
    protected static $paymentGatewayClasses = [
        GoDaddyPaymentsGateway::class,
    ];

    /** @var AbstractPaymentGateway[] */
    private static $paymentGateways = [];

    /**
     * CorePaymentGateways constructor.
     *
     * @throws Exception
     */
    public function __construct()
    {
        $this->addHooks();

        if (Onboarding::STATUS_CONNECTED === Onboarding::getStatus()) {
            static::$paymentGatewayClasses[] = GoDaddyPayInPersonGateway::class;
        }
    }

    /**
     * Loads the payments handlers.
     *
     * @internal callback
     * @see CorePaymentGateways::addHooks()
     *
     * @throws Exception
     */
    public function loadHandlers()
    {
        // don't load anything if we don't have any gateways enabled
        if (empty(static::getPaymentGateways())) {
            return;
        }

        foreach ($this->handlerClasses as $class) {
            new $class();
        }

        // TODO: load these as components once this class itself uses HasComponentsTrait {cwiseman 2021-10-21}
        (new OnboardingEventsProducer())->load();
        (new PaymentGatewayEventsProducer())->load();
        (new API())->load();
    }

    /**
     * Adds instances of the gateways contained in this class to WooCommerce gateways.
     *
     * @internal callback
     * @see CorePaymentGateways::addHooks()
     *
     * @param array $gateways
     * @return array
     * @throws Exception
     */
    public function loadPaymentGateways($gateways) : array
    {
        if (! ArrayHelper::accessible($gateways)) {
            return $gateways;
        }

        // @NOTE: Show GDP items on the top of the list
        $gd_gateways = ArrayHelper::wrap(static::getPaymentGateways());

        if ($gd_sip_gateway = ArrayHelper::get($gd_gateways, 'godaddy-payments-payinperson')) {
            // @NOTE: Move PayInPerson to the top of the list
            $gd_gateways = $gd_gateways + ['godaddy-payments-payinperson' => $gd_sip_gateway];
        }

        return array_unique($gd_gateways + $gateways, SORT_REGULAR);
    }

    /**
     * Registers the `woocommerce_payment_gateways` hook with loadPaymentGateways as the callback.
     *
     * @throws Exception
     */
    private function addHooks()
    {
        Register::action()
            ->setGroup('init')
            ->setHandler([$this, 'loadHandlers'])
            ->execute();

        Register::filter()
            ->setGroup('woocommerce_payment_gateways')
            ->setArgumentsCount(1)
            ->setHandler([$this, 'loadPaymentGateways'])
            ->execute();
    }

    /**
     * Broadcasts an event once the GDP gateway is active (available to be setup) for the first time.
     *
     * @throws Exception
     */
    protected static function maybeBroadcastPaymentGatewayFirstActiveEvent(AbstractPaymentGateway $gateway)
    {
        if (! Configuration::get('woocommerce.flags.broadcastGoDaddyPaymentsFirstActiveEvent')) {
            return;
        }

        Events::broadcast(new PaymentGatewayFirstActiveEvent($gateway->id));

        Configuration::set('woocommerce.flags.broadcastGoDaddyPaymentsFirstActiveEvent', false);

        update_option('gd_mwc_broadcast_go_daddy_payments_first_active', 'no');
    }

    /**
     * Gets a list of initialized core payment gateways.
     *
     * @return AbstractPaymentGateway[]
     * @throws Exception
     */
    public static function getPaymentGateways() : array
    {
        if (! empty(static::$paymentGateways)) {
            return static::$paymentGateways;
        }

        foreach (static::$paymentGatewayClasses as $class) {
            /** @var AbstractPaymentGateway $class */
            if (! $class::isActive()) {
                continue;
            }

            /** @var AbstractPaymentGateway $gateway */
            $gateway = new $class;

            static::$paymentGateways[$gateway->id] = $gateway;

            static::maybeBroadcastPaymentGatewayFirstActiveEvent($gateway);
        }

        return static::$paymentGateways;
    }

    /**
     * Determines whether a gateway is a platform managed payment gateway, by ID.
     *
     * @param string $gatewayId
     * @return bool
     * @throws Exception
     */
    public static function isManagedPaymentGateway(string $gatewayId) : bool
    {
        return ArrayHelper::has(static::getPaymentGateways(), $gatewayId);
    }

    /**
     * Gets an instance of a platform managed payment gateway, for a given ID.
     *
     * @param string $gatewayId
     * @return AbstractPaymentGateway|null
     * @throws Exception
     */
    public static function getManagedPaymentGatewayInstance(string $gatewayId)
    {
        $gateway = ArrayHelper::get(static::getPaymentGateways(), $gatewayId, '');

        if (empty($gateway)) {
            return null;
        }

        if (is_object($gateway)) {
            return $gateway;
        }

        return new $gateway();
    }

    /**
     * Determines that the feature can be loaded if WooCommerce is available.
     *
     * @return bool
     * @throws Exception
     */
    public static function shouldLoadConditionalFeature() : bool
    {
        return WooCommerceRepository::isWooCommerceActive();
    }
}
