<?php

namespace GoDaddy\WordPress\MWC\Core\Payments\Poynt\Events\Subscribers;

use Exception;
use GoDaddy\WordPress\MWC\Common\Components\Contracts\ComponentContract;
use GoDaddy\WordPress\MWC\Common\Configuration\Configuration;
use GoDaddy\WordPress\MWC\Common\DataSources\WooCommerce\Adapters\CurrencyAmountAdapter;
use GoDaddy\WordPress\MWC\Common\Events\Contracts\EventContract;
use GoDaddy\WordPress\MWC\Common\Events\Events;
use GoDaddy\WordPress\MWC\Common\Helpers\ArrayHelper;
use GoDaddy\WordPress\MWC\Common\Helpers\StringHelper;
use GoDaddy\WordPress\MWC\Common\Http\Response;
use GoDaddy\WordPress\MWC\Common\Models\CurrencyAmount;
use GoDaddy\WordPress\MWC\Common\Register\Register;
use GoDaddy\WordPress\MWC\Common\Repositories\WooCommerce\OrdersRepository;
use GoDaddy\WordPress\MWC\Core\Events\AbstractWebhookReceivedEvent;
use GoDaddy\WordPress\MWC\Core\Events\BeforeCreateRefundEvent;
use GoDaddy\WordPress\MWC\Core\Events\BeforeCreateVoidEvent;
use GoDaddy\WordPress\MWC\Core\Events\CaptureTransactionEvent;
use GoDaddy\WordPress\MWC\Core\Events\Subscribers\AbstractWebhookReceivedSubscriber;
use GoDaddy\WordPress\MWC\Core\Exceptions\Payments\PoyntWebhookInvalidSignatureException;
use GoDaddy\WordPress\MWC\Core\Payments\Adapters\OrderAdapter;
use GoDaddy\WordPress\MWC\Core\Payments\DataStores\WooCommerce\OrderCaptureTransactionDataStore;
use GoDaddy\WordPress\MWC\Core\Payments\DataStores\WooCommerce\OrderPaymentTransactionDataStore;
use GoDaddy\WordPress\MWC\Core\Payments\DataStores\WooCommerce\OrderTransactionDataStore;
use GoDaddy\WordPress\MWC\Core\Payments\Models\Transactions\PaymentTransaction;
use GoDaddy\WordPress\MWC\Core\Payments\Poynt;
use GoDaddy\WordPress\MWC\Core\Payments\Poynt\Events\Producers\PushOrdersProducer;
use GoDaddy\WordPress\MWC\Core\Payments\Poynt\Http\Adapters\AuthorizationTransactionAdapter;
use GoDaddy\WordPress\MWC\Core\Payments\Poynt\Http\Adapters\CaptureTransactionAdapter;
use GoDaddy\WordPress\MWC\Core\Payments\Poynt\Http\Adapters\PaymentTransactionAdapter;
use GoDaddy\WordPress\MWC\Core\Payments\Poynt\Http\Adapters\RefundTransactionAdapter;
use GoDaddy\WordPress\MWC\Core\Payments\Poynt\Http\Adapters\VoidTransactionAdapter;
use GoDaddy\WordPress\MWC\Core\Payments\Poynt\Http\GetOrderRequest;
use GoDaddy\WordPress\MWC\Core\Payments\Poynt\Http\GetTransactionRequest;
use GoDaddy\WordPress\MWC\Core\WooCommerce\Emails\ReadyForPickupEmail;
use GoDaddy\WordPress\MWC\Core\WooCommerce\Payments\CorePaymentGateways;
use GoDaddy\WordPress\MWC\Payments\Events\PaymentTransactionEvent;
use GoDaddy\WordPress\MWC\Payments\Events\RefundTransactionEvent;
use GoDaddy\WordPress\MWC\Payments\Events\VoidTransactionEvent;
use GoDaddy\WordPress\MWC\Payments\Models\Transactions\AbstractTransaction;
use GoDaddy\WordPress\MWC\Payments\Models\Transactions\AuthorizationTransaction;
use GoDaddy\WordPress\MWC\Payments\Models\Transactions\CaptureTransaction;
use GoDaddy\WordPress\MWC\Payments\Models\Transactions\RefundTransaction;
use GoDaddy\WordPress\MWC\Payments\Models\Transactions\VoidTransaction;
use WC_Order;
use WC_Order_Item_Fee;

/**
 * TODO: consider as a possible refactor, extracting the logic that deals with Terminal-originated transactions, and handling in a PaymentGateway implementation. e.g. RemotePaymentGateway, or OfflinePaymentGateway, or DevicePaymentGateway {JS - 2021-10-16}.
 */
class WebhookSubscriber extends AbstractWebhookReceivedSubscriber implements ComponentContract
{
    /** @var string order cancelled webhook event type */
    const ORDER_CANCELLED_EVENT_TYPE = 'ORDER_CANCELLED';

    /** @var string order completed webhook event type */
    const ORDER_COMPLETED_EVENT_TYPE = 'ORDER_COMPLETED';

    /** @var string order updated webhook event type */
    const ORDER_UPDATED_EVENT_TYPE = 'ORDER_UPDATED';

    /** @var string transaction captured webhook event type */
    const TRANSACTION_CAPTURED_EVENT_TYPE = 'TRANSACTION_CAPTURED';

    /** @var string transaction authorized webhook event type */
    const TRANSACTION_AUTHORIZED_EVENT_TYPE = 'TRANSACTION_AUTHORIZED';

    /** @var string transaction refunded webhook event type */
    const TRANSACTION_REFUNDED_EVENT_TYPE = 'TRANSACTION_REFUNDED';

    /** @var string transaction voided webhook event type */
    const TRANSACTION_VOIDED_EVENT_TYPE = 'TRANSACTION_VOIDED';

    /** @var string resource value for transactions webhooks */
    const EVENT_RESOURCE_TRANSACTIONS = '/transactions';

    /** @var string resource value for orders webhooks */
    const EVENT_RESOURCE_ORDERS = '/orders';

    /** @var string */
    private $remoteOrderId;

    /**
     * Required for ComponentContract interface.
     */
    public function load()
    {
        // no-op
    }

    /**
     * Determines whether the event should be handled.
     *
     * @param EventContract $event
     * @return bool
     * @throws Exception
     */
    public function shouldHandle(EventContract $event) : bool
    {
        return Configuration::get('features.bopit', false) && parent::shouldHandle($event);
    }

    /**
     * Validates a webhook.
     *
     * @param AbstractWebhookReceivedEvent $event
     * @return bool
     * @throws Exception
     */
    public function validate(AbstractWebhookReceivedEvent $event) : bool
    {
        $signature = base64_encode(hash_hmac('sha1', $event->getPayload(), Poynt::getWebhookSecret(), true));

        if (! hash_equals($signature, ArrayHelper::get($event->getHeaders(), 'HTTP_POYNT_WEBHOOK_SIGNATURE'))) {
            throw new PoyntWebhookInvalidSignatureException('Invalid webhook signature');
        }

        return StringHelper::isJson($event->getPayload());
    }

    /**
     * Handles the event payload.
     *
     * @param AbstractWebhookReceivedEvent $event
     * @throws Exception
     */
    public function handlePayload(AbstractWebhookReceivedEvent $event)
    {
        $data = $event->getPayloadDecoded();
        $eventType = ArrayHelper::get($data, 'eventType');
        $eventResource = ArrayHelper::get($data, 'resource');

        if (static::EVENT_RESOURCE_TRANSACTIONS === $eventResource && ! $transaction = $this->fetchRemoteTransaction($event)) {
            return;
        }

        if (! $wcOrder = $this->findOrderForPayload($data)) {
            return;
        }

        if (isset($transaction) && ! $transaction->getProviderName()) {
            $transaction->setProviderName(OrderTransactionDataStore::readProviderName($wcOrder->get_id()));
        }

        if (static::ORDER_CANCELLED_EVENT_TYPE === $eventType) {
            $this->handleOrderCancelledEvent($wcOrder);
        } elseif (static::ORDER_COMPLETED_EVENT_TYPE === $eventType) {
            $this->handleOrderCompletedEvent($wcOrder);
        } elseif (static::ORDER_UPDATED_EVENT_TYPE === $eventType) {
            $this->handleOrderUpdatedEvent($data, $wcOrder);
        } elseif (static::TRANSACTION_CAPTURED_EVENT_TYPE === $eventType && $transaction instanceof CaptureTransaction) {
            $this->handleTransactionCapturedEvent($transaction, $wcOrder);
        } elseif (static::TRANSACTION_REFUNDED_EVENT_TYPE === $eventType) {
            $this->handleTransactionRefundedEvent($transaction, $wcOrder);
        } elseif (static::TRANSACTION_CAPTURED_EVENT_TYPE === $eventType && $transaction instanceof PaymentTransaction) {
            $this->handleTransactionSaleEvent($transaction, $wcOrder);
        } elseif (static::TRANSACTION_VOIDED_EVENT_TYPE === $eventType && $transaction instanceof VoidTransaction) {
            $this->handleTransactionVoidedEvent($transaction, $wcOrder);
        } elseif (static::TRANSACTION_AUTHORIZED_EVENT_TYPE === $eventType && $transaction instanceof PaymentTransaction) {
            $this->handleTransactionAuthorizedEvent($transaction, $wcOrder);
        }
    }

    /**
     * Fetch and return the remote transaction object.
     *
     * @return AbstractTransaction|null
     * @return PaymentTransaction|CaptureTransaction|RefundTransaction|VoidTransaction
     * @throws Exception
     */
    protected function fetchRemoteTransaction(AbstractWebhookReceivedEvent $event)
    {
        $data = $event->getPayloadDecoded();
        $eventType = ArrayHelper::get($data, 'eventType');
        $resourceId = ArrayHelper::get($data, 'resourceId');

        $transactionResponse = $this->getRemoteTransactionResponse($resourceId);

        $remoteOrderReference = current(ArrayHelper::where(ArrayHelper::get($transactionResponse->getBody(), 'references', []), function ($value) {
            return 'POYNT_ORDER' === ArrayHelper::get($value, 'type');
        }));

        // ugh, this side effect pains me, but it's late and I don't have a more elegant idea right now {JS - 2021-10-15}
        $this->remoteOrderId = ArrayHelper::get($remoteOrderReference, 'id');

        // capture and void transactions operate on an existing authorization
        // transaction, and it's that authorization transaction that fires the
        // webhook. So first we grab that Auth transaction, and then we grab
        // the Capture or Void transaction.
        if (static::TRANSACTION_CAPTURED_EVENT_TYPE === $eventType) {
            if ('AUTHORIZE' === ArrayHelper::get($transactionResponse->getBody(), 'action')) {
                $authTransaction = (new AuthorizationTransactionAdapter(new AuthorizationTransaction()))
                    ->convertToSource($transactionResponse);

                $captureTransactionResponse = $this->getRemoteTransactionResponse($authTransaction->getRemoteCaptureId());
                $captureTransaction = (new CaptureTransactionAdapter(new CaptureTransaction()))
                    ->convertToSource($captureTransactionResponse)
                    ->setProviderName('poynt');

                // correct for an issue in getTransactionRemoteId
                $captureTransaction->setRemoteId($authTransaction->getRemoteCaptureId());

                return $captureTransaction;
            } elseif ('SALE' === ArrayHelper::get($transactionResponse->getBody(), 'action')) {
                $saleTransaction = (new PaymentTransactionAdapter(new PaymentTransaction()))
                    ->convertToSource($transactionResponse)
                    ->setProviderName('poynt');

                return $saleTransaction;
            } elseif ('CAPTURE' === ArrayHelper::get($transactionResponse->getBody(), 'action')) {
                $captureTransaction = (new CaptureTransactionAdapter(new CaptureTransaction()))
                    ->convertToSource($transactionResponse)
                    ->setProviderName('poynt');

                // correct for an issue in getTransactionRemoteId
                $captureTransaction->setRemoteId($resourceId);

                return $captureTransaction;
            }
        } elseif ($eventType === static::TRANSACTION_REFUNDED_EVENT_TYPE) {
            return (new RefundTransactionAdapter(new RefundTransaction()))
                ->convertToSource($transactionResponse);
        } elseif (static::TRANSACTION_VOIDED_EVENT_TYPE === $eventType) {
            // get void transaction ID from the authorization transaction data
            $responseBody = $transactionResponse->getBody() ?? [];

            $voidTransactionLink = current(ArrayHelper::where(ArrayHelper::get($responseBody, 'links', []), function ($value) {
                return 'REFUND' === ArrayHelper::get($value, 'rel');
            }));
            $voidTransactionId = ArrayHelper::get($voidTransactionLink, 'href');

            // fetch void transaction from Poynt API
            $voidTransactionResponse = $this->getRemoteTransactionResponse($voidTransactionId);

            return (new VoidTransactionAdapter(new VoidTransaction()))
                ->convertToSource($voidTransactionResponse);
        } elseif ($eventType === static::TRANSACTION_AUTHORIZED_EVENT_TYPE) {
            return (new PaymentTransactionAdapter(new PaymentTransaction()))
                    ->convertToSource($transactionResponse)
                    ->setProviderName('poynt');
        }
    }

    /**
     * Find the corresponding WC Order based on the webhook payload data.
     *
     * @return WC_Order|null
     * @throws Exception
     */
    protected function findOrderForPayload(array $data)
    {
        $poyntOrderId = null;
        $resourceId = ArrayHelper::get($data, 'resourceId');
        $resource = ArrayHelper::get($data, 'resource');

        if ($resource === static::EVENT_RESOURCE_ORDERS) {
            $poyntOrderId = $resourceId;
        } elseif ($resource === static::EVENT_RESOURCE_TRANSACTIONS) {
            $poyntOrderId = $this->remoteOrderId;
        }

        if ($poyntOrderId) {
            $args = [
                'post_type'   => 'shop_order',
                'fields'      => 'ids',
                'post_status' => 'any',
                'meta_key'    => '_poynt_order_remoteId',
                'meta_value'  => $poyntOrderId,
            ];

            if ($results = get_posts($args)) {
                return OrdersRepository::get($results[0]);
            }
        }

        return null;
    }

    /**
     * Handles an order cancelled event.
     *
     * @param WC_Order $wcOrder
     */
    protected function handleOrderCancelledEvent(WC_Order $wcOrder)
    {
        if ('refunded' !== $wcOrder->get_status()) {
            $wcOrder->update_status('cancelled');
        }
    }

    /**
     * Handles an order completed event.
     *
     * @param WC_Order $wcOrder
     */
    protected function handleOrderCompletedEvent(WC_Order $wcOrder)
    {
        if (! ArrayHelper::contains(['refunded', 'cancelled'], $wcOrder->get_status())) {
            $wcOrder->update_status('completed');
        }
    }

    /**
     * Handles an order updated event.
     *
     * @param array $data
     * @param WC_Order $wcOrder
     */
    protected function handleOrderUpdatedEvent(array $data, WC_Order $wcOrder)
    {
        $this->maybeHandleOrderReadyForPickup($data, $wcOrder);
    }

    /**
     * Handles a transaction capture event.
     *
     * @param CaptureTransaction $transaction
     * @param WC_Order $wcOrder
     * @throws Exception
     */
    protected function handleTransactionCapturedEvent(CaptureTransaction $transaction, WC_Order $wcOrder)
    {
        $transaction->setOrder((new OrderAdapter($wcOrder))->convertFromSource());

        if ($transaction->getOrder()->isCaptured()) {
            // capture event has already been processed, so bail
            return;
        }

        $this->addOrderItemFees($transaction, $wcOrder);

        // persist the capture meta to the datastore
        (new OrderCaptureTransactionDataStore($transaction->getProviderName()))->save($transaction);

        Events::broadcast(new CaptureTransactionEvent($transaction));

        if ('on-hold' === $wcOrder->get_status()) {
            $wcOrder->update_status('processing');
        }
    }

    /**
     * Handles a transaction refund event. There are a variety of cases we need
     * to consider:.
     *
     * - webhook from a GDP processed refund originating from WC: skip processing
     * - webhook from a 3rd party gateway / manual refund originating from WC: skip processing
     * - webhook from the Smart Terminal due to a GDP processed refund: mark the WC order as refunded
     * - webhook from the Smart Terminal due to a 3rd party gateway refund action: perform the 3rd party gateway refund in WC
     * - duplicate / replay of webhook of any of the above cases: skip processing
     *
     * @param RefundTransaction $transaction
     * @param WC_Order $wcOrder
     * @throws Exception
     */
    protected function handleTransactionRefundedEvent(RefundTransaction $transaction, WC_Order $wcOrder)
    {
        $transaction->setOrder((new OrderAdapter($wcOrder))->convertFromSource());

        // remote refund has already been processed: skip
        if ($wcOrder->get_meta('_poynt_refund_remoteId')) {
            return;
        }

        if ($transaction->getProviderName() != $wcOrder->get_payment_method()) {
            // tell WC that we want to process this refund with the transaction provider gateway,
            // regardless of the actual gateway used (e.g. Cash on Delivery)
            Register::filter()
                ->setGroup('woocommerce_order_get_payment_method')
                ->setHandler(function () use ($transaction) {
                    return $transaction->getProviderName();
                })
                ->execute();
        }

        Events::broadcast(new BeforeCreateRefundEvent($transaction));

        // perform a full refund, re-stocking all items
        wc_create_refund([
            'amount'          => $wcOrder->get_total(),
            'reason'          => __('Refund from Smart Terminal', 'mwc-core'),
            'order_id'        => $wcOrder->get_id(),
            'refund_payment'  => true,
            'restock_items'   => true,
            'line_items'      => $this->lineItemsToSimpleArray($wcOrder->get_items(['line_item', 'fee', 'shipping'])),
            'skip_bopit_sync' => true,
        ]);

        $wcOrder->update_meta_data('_poynt_refund_remoteId', $transaction->getRemoteId());
        $wcOrder->save();
    }

    /**
     * Converts WC order item objects (line_item, shipping, and fee).
     *
     * Formats the items as used by {@see wc_create_refund()}.
     *
     * @param array $lineItems
     * @return array
     */
    private function lineItemsToSimpleArray(array $lineItems) : array
    {
        $result = [];

        foreach ($lineItems as $id => $item) {
            // should we be using totals or subtotals here?
            $result[$id] = [
                'qty'          => $item->get_type() == 'line_item' ? $item->get_quantity() : 0,
                'refund_total' => $item->get_total(),
                'refund_tax'   => ArrayHelper::get($item->get_taxes(), 'total'),
            ];
        }

        return $result;
    }

    /**
     * Handles a transaction sale (capture) event.
     *
     * @param PaymentTransaction $transaction
     * @param WC_Order $wcOrder
     * @throws Exception
     */
    protected function handleTransactionSaleEvent(PaymentTransaction $transaction, WC_Order $wcOrder)
    {
        $transaction->setOrder((new OrderAdapter($wcOrder))->convertFromSource());

        if ($transaction->getOrder()->isCaptured()) {
            // sale event has already been processed, so bail
            return;
        }

        $this->addOrderItemFees($transaction, $wcOrder);

        // persist the sale meta to the datastore
        (new OrderPaymentTransactionDataStore($transaction->getProviderName()))->save($transaction);

        $wcOrder->payment_complete($transaction->getRemoteId());

        Events::broadcast(new PaymentTransactionEvent($transaction));
    }

    /**
     * Handles a transaction void event.
     *
     * @param VoidTransaction $transaction
     * @param WC_Order $wcOrder
     * @throws Exception
     */
    protected function handleTransactionVoidedEvent(VoidTransaction $transaction, WC_Order $wcOrder)
    {
        $transaction->setOrder((new OrderAdapter($wcOrder))->convertFromSource());

        // remote void has already been processed: skip
        if ($wcOrder->get_meta('_poynt_void_remoteId')) {
            return;
        }

        if ($transaction->getProviderName() != $wcOrder->get_payment_method()) {
            // tell WC that we want to process this void with the transaction provider gateway,
            // regardless of the actual gateway used (e.g. Cash on Delivery)
            Register::filter()
                ->setGroup('woocommerce_order_get_payment_method')
                ->setHandler(function () use ($transaction) {
                    return $transaction->getProviderName();
                })
                ->execute();
        }

        Events::broadcast(new BeforeCreateVoidEvent($transaction));

        // perform a full refund, re-stocking all items
        wc_create_refund([
            'amount'          => $wcOrder->get_total(),
            'reason'          => __('Void from Smart Terminal', 'mwc-core'),
            'order_id'        => $wcOrder->get_id(),
            'refund_payment'  => true,
            'restock_items'   => true,
            'line_items'      => $this->lineItemsToSimpleArray($wcOrder->get_items(['line_item', 'fee', 'shipping'])),
            'skip_bopit_sync' => true,
        ]);

        $wcOrder->update_meta_data('_poynt_void_remoteId', $transaction->getRemoteId());
        $wcOrder->save();
    }

    /**
     * Add order meta and order notes on an AUTHORIZED transaction webhook response.
     *
     * @param PaymentTransaction $transaction
     * @param WC_Order $wcOrder
     * @throws Exception
     */
    protected function handleTransactionAuthorizedEvent(PaymentTransaction $transaction, WC_Order $wcOrder)
    {
        $transaction->setOrder((new OrderAdapter($wcOrder))->convertFromSource());

        if ((new OrderPaymentTransactionDataStore('poynt'))->read($transaction->getOrder()->getId(), 'payment')->getRemoteId()) {
            // authorization event has already been processed
            return;
        }

        $this->addOrderItemFees($transaction, $wcOrder);

        // persist the authorization meta to the datastore
        (new OrderPaymentTransactionDataStore($transaction->getProviderName()))->save($transaction);

        if ('on-hold' === $wcOrder->get_status()) {
            $wcOrder->update_status('processing');
        }

        // broadcast the payment event to relevant add order notes
        Events::broadcast(new PaymentTransactionEvent($transaction));
    }

    /**
     * Gets the remote transaction from Poynt API.
     *
     * @param string $transactionId
     * @return Response
     * @throws Exception
     */
    protected function getRemoteTransactionResponse(string $transactionId) : Response
    {
        $response = (new GetTransactionRequest($transactionId))->send();

        if ($response->isError() || $response->getStatus() !== 200) {
            $errorMessage = ArrayHelper::get($response->getBody(), 'developerMessage');
            throw new Exception("Could not retrieve transaction {$transactionId} from Poynt ({$response->getStatus()}): {$errorMessage}");
        }

        return $response;
    }

    /**
     * Gets the remote order data from Poynt API.
     *
     * @param array $data
     * @return array
     * @throws Exception
     */
    protected function getRemoteOrderData(array $data) : array
    {
        $poyntOrderId = ArrayHelper::get($data, 'resourceId');
        $response = (new GetOrderRequest($poyntOrderId))->send();

        if ($response->isError() || $response->getStatus() !== 200) {
            $errorMessage = ArrayHelper::get($response->getBody(), 'developerMessage');
            throw new Exception("Could not retrieve order {$poyntOrderId} from Poynt ({$response->getStatus()}): {$errorMessage}");
        }

        if ($response->isSuccess()) {
            return $response->getBody();
        }

        return [];
    }

    /**
     * May handle the event received when an order is marked as ready for pickup.
     *
     * @param array $data
     * @param WC_Order $wcOrder
     * @throws Exception
     */
    protected function maybeHandleOrderReadyForPickup(array $data, WC_Order $wcOrder)
    {
        if ($wcOrder->get_meta('_poynt_order_status_ready_at')) {
            return;
        }

        if (! empty($orderData = $this->getRemoteOrderData($data)) && $this->isOrderReadyForPickup($orderData)) {
            $this->handleOrderReadyForPickup($orderData, $wcOrder);
        }
    }

    /**
     * Checks if the order is ready for pickup.
     *
     * @param array $orderData from response body
     * @return bool
     */
    protected function isOrderReadyForPickup(array $orderData) : bool
    {
        if ('OPENED' != ArrayHelper::get($orderData, 'statuses.status')) {
            return false;
        }

        foreach (ArrayHelper::get($orderData, 'orderShipments', []) as $shipment) {
            if ('PICKUP' === ArrayHelper::get($shipment, 'deliveryMode')
                && 'AWAITING_PICKUP' === ArrayHelper::get($shipment, 'status')) {
                return true;
            }
        }

        return false;
    }

    /**
     * Handles communication when an order is marked as ready for pickup.
     *
     * @param array $orderData from response body
     * @param WC_Order $wcOrder
     * @throws Exception
     */
    protected function handleOrderReadyForPickup(array $orderData, WC_Order $wcOrder)
    {
        if ($wcOrder->get_meta('_poynt_order_status_ready_at')) {
            return;
        }

        $readyAtEvent = current(ArrayHelper::where(ArrayHelper::get($orderData, 'orderHistories', []), function ($value) {
            return 'AWAITING_PICKUP' === ArrayHelper::get($value, 'event');
        }));

        $readyAt = ArrayHelper::get($readyAtEvent, 'timestamp') ?: ArrayHelper::get($orderData, 'updatedAt');

        $wcOrder->add_meta_data('_poynt_order_status_ready_at', $readyAt);
        $wcOrder->save_meta_data();

        $wcOrder->add_order_note(sprintf(
             /* translators: Placeholders: %1$s - date, %2$s time */
            __('Order marked ready on terminal on %1$s at %2$s', 'mwc-core'),
            wc_format_datetime(new \WC_DateTime($readyAt)),
            wc_format_datetime(new \WC_DateTime($readyAt), wc_time_format())
        ));

        (new ReadyForPickupEmail())->trigger($wcOrder->get_id(), $wcOrder);
    }

    /**
     * May add order item fees.
     *
     * @param CaptureTransaction|PaymentTransaction $transaction
     * @param WC_Order $wcOrder
     */
    protected function addOrderItemFees(AbstractTransaction $transaction, WC_Order $wcOrder)
    {
        $shouldCalculateTotals = false;

        // may add a tip amount, if any
        if (is_callable([$transaction, 'getTipAmount']) && ! empty($tipAmount = $transaction->getTipAmount())) {
            $shouldCalculateTotals = $this->addOrderItemFee(__('Tip', 'mwc-core'), $tipAmount, $wcOrder);
        }

        // may add a cashback amount, if any
        if (is_callable([$transaction, 'getCashbackAmount']) && ! empty($cashbackAmount = $transaction->getCashbackAmount())) {
            $shouldCalculateTotals = $this->addOrderItemFee(__('Cashback', 'mwc-core'), $cashbackAmount, $wcOrder) || $shouldCalculateTotals;
        }

        if ($shouldCalculateTotals) {
            $wcOrder->calculate_totals();
        }
    }

    /**
     * Adds an order item fee to an order.
     *
     * This method can be used to add items from a transaction like a tip or a cashback.
     *
     * @param string $itemFeeName
     * @param CurrencyAmount $amount
     * @param WC_Order $order
     * @return bool
     */
    protected function addOrderItemFee(string $itemFeeName, CurrencyAmount $amount, WC_Order $order) : bool
    {
        if (0 === $amount->getAmount() || $this->orderHasItemFee($order, $itemFeeName)) {
            return false;
        }

        $convertedAmount = (new CurrencyAmountAdapter(0, ''))->convertToSource($amount);

        $item = $this->createOrderItemFee($convertedAmount, $itemFeeName);

        $order->add_item($item);
        $order->add_order_note(sprintf(
            /* translators: Placeholders: %1$s - item fee name, %2$s - item fee amount */
            __('%1$s amount of %2$s added to order by GoDaddy Payments Smart Terminal', 'mwc-core'),
            $itemFeeName,
            wc_price($convertedAmount, get_woocommerce_currency_symbol())
        ));

        return true;
    }

    /**
     * Creates a new WC_Order_Item_Fee with the provided amount.
     *
     * All fees created here are non-taxable and must be used to add order items like tip, cashback, etc.
     *
     * @param float $amount
     * @param string $feeName
     * @return WC_Order_Item_Fee
     */
    protected function createOrderItemFee(float $amount, string $feeName) : WC_Order_Item_Fee
    {
        $item = new WC_Order_Item_Fee();
        $item->set_name($feeName);
        $item->set_amount($amount);
        $item->set_total($amount);
        $item->set_tax_status('none');

        return $item;
    }

    /**
     * Determines whether an order has a item fee with a specific name.
     *
     * @param WC_Order $order
     * @param string $itemFeeName
     *
     * @return bool
     */
    protected function orderHasItemFee(WC_Order $order, string $itemFeeName) : bool
    {
        foreach ($order->get_fees() as $item) {
            if ($item instanceof WC_Order_Item_Fee && $itemFeeName === $item->get_name()) {
                return true;
            }
        }

        return false;
    }
}
