<?php

namespace GoDaddy\WordPress\MWC\Core\Payments\Poynt\Events\Subscribers;

use Exception;
use GoDaddy\WordPress\MWC\Common\DataSources\WooCommerce\Adapters\CurrencyAmountAdapter;
use GoDaddy\WordPress\MWC\Common\Events\Contracts\EventContract;
use GoDaddy\WordPress\MWC\Common\Events\Contracts\SubscriberContract;
use GoDaddy\WordPress\MWC\Common\Events\Events;
use GoDaddy\WordPress\MWC\Common\Helpers\ArrayHelper;
use GoDaddy\WordPress\MWC\Common\Models\CurrencyAmount;
use GoDaddy\WordPress\MWC\Common\Repositories\WooCommerce\OrdersRepository;
use GoDaddy\WordPress\MWC\Core\Payments\DataStores\WooCommerce\OrderCaptureTransactionDataStore;
use GoDaddy\WordPress\MWC\Core\Payments\DataStores\WooCommerce\OrderPaymentTransactionDataStore;
use GoDaddy\WordPress\MWC\Core\Payments\Models\Orders\Order;
use GoDaddy\WordPress\MWC\Core\Payments\Models\Transactions\PaymentTransaction;
use GoDaddy\WordPress\MWC\Core\Payments\Poynt\Events\WebhookReceivedEvent;
use GoDaddy\WordPress\MWC\Payments\Events\CaptureTransactionEvent;
use GoDaddy\WordPress\MWC\Payments\Events\PaymentTransactionEvent;
use GoDaddy\WordPress\MWC\Payments\Models\Transactions\AbstractTransaction;
use GoDaddy\WordPress\MWC\Payments\Models\Transactions\CaptureTransaction;
use GoDaddy\WordPress\MWC\Payments\Models\Transactions\RefundTransaction;
use GoDaddy\WordPress\MWC\Payments\Models\Transactions\VoidTransaction;
use WC_Order;
use WC_Order_Item_Fee;

/**
 * An events' subscriber for transaction-specific webhooks.
 */
class TransactionWebhookReceivedSubscriber implements SubscriberContract
{
    /** @var string transaction captured webhook event type */
    const TRANSACTION_CAPTURED_EVENT_TYPE = 'TRANSACTION_CAPTURED';

    /** @var string transaction authorized webhook event type */
    const TRANSACTION_AUTHORIZED_EVENT_TYPE = 'TRANSACTION_AUTHORIZED';

    /** @var string transaction refunded webhook event type */
    const TRANSACTION_REFUNDED_EVENT_TYPE = 'TRANSACTION_REFUNDED';

    /** @var string transaction voided webhook event type */
    const TRANSACTION_VOIDED_EVENT_TYPE = 'TRANSACTION_VOIDED';

    /**
     * Handles the transaction received event.
     *
     * @param EventContract $event
     * @return void
     * @throws Exception
     */
    public function handle(EventContract $event)
    {
        // only handle Poynt webhook received events
        if (! is_a($event, WebhookReceivedEvent::class)) {
            return;
        }

        // only proceed if a transaction is found
        if (! $transaction = $this->fetchRemoteTransaction($event)) {
            return;
        }

        $this->handleEventType((string) ArrayHelper::get($event->getPayloadDecoded(), 'eventType', ''), $transaction);
    }

    /**
     * Handles the given event type and transaction.
     *
     * @param string $eventType
     * @param AbstractTransaction $transaction
     * @throws Exception
     */
    protected function handleEventType(string $eventType, AbstractTransaction $transaction)
    {
        switch ($eventType) {
            case static::TRANSACTION_AUTHORIZED_EVENT_TYPE:
                if ($transaction instanceof PaymentTransaction) {
                    $this->handleTransactionAuthorizedEvent($transaction);
                }
                break;
            case static::TRANSACTION_CAPTURED_EVENT_TYPE:
                if ($transaction instanceof CaptureTransaction) {
                    $this->handleTransactionCapturedEvent($transaction);
                } elseif ($transaction instanceof PaymentTransaction) {
                    $this->handleTransactionSaleEvent($transaction);
                }
                break;
            case static::TRANSACTION_REFUNDED_EVENT_TYPE:
                if ($transaction instanceof RefundTransaction) {
                    $this->handleTransactionRefundedEvent($transaction);
                }
                break;
            case static::TRANSACTION_VOIDED_EVENT_TYPE:
                if ($transaction instanceof VoidTransaction) {
                    $this->handleTransactionVoidedEvent($transaction);
                }
                break;
        }
    }

    /**
     * Gets the remote transaction for a webhook received event.
     *
     * @return AbstractTransaction|null
     */
    public function fetchRemoteTransaction(WebhookReceivedEvent $event)
    {
        // @TODO implement this method MWC-3108 {unfulvio 2021-11-02}
        return null;
    }

    /**
     * Gets the transaction order.
     *
     * @return Order|null
     */
    protected function getTransactionOrder()
    {
        // @TODO implement this method MWC-3106 {unfulvio 2021-11-02}
        return null;
    }

    /**
     * Handles a transaction capture event.
     *
     * @param CaptureTransaction $transaction
     * @throws Exception
     */
    protected function handleTransactionCapturedEvent(CaptureTransaction $transaction)
    {
        /** @var Order $order */
        $order = $transaction->getOrder();

        // bail if the transaction capture event has already been processed
        if ($order->isCaptured()) {
            return;
        }

        $wcOrder = OrdersRepository::get($order->getId());

        if (! $wcOrder) {
            return;
        }

        $this->addOrderItemFees($transaction, $wcOrder);

        $this->getOrderCaptureTransactionDataStoreForProvider($transaction->getProviderName())->save($transaction);

        Events::broadcast($this->getNewCaptureTransactionEvent($transaction));

        if ('on-hold' === $wcOrder->get_status()) {
            $wcOrder->update_status('processing');
        }
    }

    /**
     * Gets the order capture transaction data store for a given provider.
     *
     * @param string $providerName
     * @return OrderCaptureTransactionDataStore
     */
    protected function getOrderCaptureTransactionDataStoreForProvider(string $providerName) : OrderCaptureTransactionDataStore
    {
        return new OrderCaptureTransactionDataStore($providerName);
    }

    /**
     * Gets a new capture transaction event for a given transaction.
     *
     * @param CaptureTransaction $transaction
     * @return CaptureTransactionEvent
     */
    protected function getNewCaptureTransactionEvent(CaptureTransaction $transaction) : CaptureTransactionEvent
    {
        return new CaptureTransactionEvent($transaction);
    }

    /**
     * Handles a transaction sale (capture) event.
     *
     * @param PaymentTransaction $transaction
     * @throws Exception
     */
    protected function handleTransactionSaleEvent(PaymentTransaction $transaction)
    {
        /** @var Order $order */
        $order = $transaction->getOrder();

        // bail if sale event (capture) has already been processed
        if ($order->isCaptured()) {
            return;
        }

        $wcOrder = OrdersRepository::get($order->getId());

        if (! $wcOrder) {
            return;
        }

        $this->addOrderItemFees($transaction, $wcOrder);

        $this->getOrderPaymentTransactionDataStoreForProvider($transaction->getProviderName())
            ->save($transaction);

        $wcOrder->payment_complete($transaction->getRemoteId());

        Events::broadcast($this->getNewPaymentTransactionEvent($transaction));
    }

    /**
     * Handles a transaction authorization event.
     *
     * @param PaymentTransaction $transaction
     * @throws Exception
     */
    protected function handleTransactionAuthorizedEvent(PaymentTransaction $transaction)
    {
        $orderId = $transaction->getOrder()->getId();
        $paymentTransaction = $this->getOrderPaymentTransactionDataStoreForProvider('poynt')
            ->read($orderId, 'payment');

        // bail if authorization event has already been processed
        if ($paymentTransaction->getRemoteId()) {
            return;
        }

        $wcOrder = OrdersRepository::get($orderId);

        if (! $wcOrder) {
            return;
        }

        $this->addOrderItemFees($transaction, $wcOrder);

        $this->getOrderPaymentTransactionDataStoreForProvider($transaction->getProviderName())
            ->save($transaction);

        if ('on-hold' === $wcOrder->get_status()) {
            $wcOrder->update_status('processing');
        }

        Events::broadcast($this->getNewPaymentTransactionEvent($transaction));
    }

    /**
     * Gets the order payment transaction data store for a given provider.
     *
     * @param string $providerName
     * @return OrderPaymentTransactionDataStore
     */
    protected function getOrderPaymentTransactionDataStoreForProvider(string $providerName) : OrderPaymentTransactionDataStore
    {
        return new OrderPaymentTransactionDataStore($providerName);
    }

    /**
     * Gets a new payment transaction event for a given transaction.
     *
     * @param PaymentTransaction $transaction
     * @return PaymentTransactionEvent
     */
    protected function getNewPaymentTransactionEvent(PaymentTransaction $transaction) : PaymentTransactionEvent
    {
        return new PaymentTransactionEvent($transaction);
    }

    /**
     * Handles a transaction refund event.
     *
     * @param RefundTransaction $transaction
     * @throws Exception
     */
    protected function handleTransactionRefundedEvent(RefundTransaction $transaction)
    {
        // TODO implement this method MWC-3104 {unfulvio 2021-11-02}
    }

    /**
     * Handles a transaction void event.
     *
     * @param VoidTransaction $transaction
     * @throws Exception
     */
    protected function handleTransactionVoidedEvent(VoidTransaction $transaction)
    {
        // @TODO implement this method MWC-3105 {unfulvio 2021-11-02}
    }

    /**
     * Adds WooCommerce order item fees based on transaction data.
     *
     * @param AbstractTransaction $transaction
     * @param WC_Order $wcOrder
     */
    protected function addOrderItemFees(AbstractTransaction $transaction, WC_Order $wcOrder)
    {
        // may add a tip amount, if any
        if (is_callable([$transaction, 'getTipAmount']) && ! empty($tipAmount = $transaction->getTipAmount())) {
            $shouldCalculateTotals = $this->addOrderItemFee(__('Tip', 'mwc-core'), $tipAmount, $wcOrder);
        }

        // may add a cashback amount, if any
        if (is_callable([$transaction, 'getCashbackAmount']) && ! empty($cashbackAmount = $transaction->getCashbackAmount())) {
            $shouldCalculateTotals = $this->addOrderItemFee(__('Cashback', 'mwc-core'), $cashbackAmount, $wcOrder) || $shouldCalculateTotals;
        }

        if (! empty($shouldCalculateTotals)) {
            $wcOrder->calculate_totals();
        }
    }

    /**
     * Adds an order item fee to an order.
     *
     * This method can be used to add items from a transaction like a tip or a cashback.
     *
     * @param string $itemFeeName
     * @param CurrencyAmount $amount
     * @param WC_Order $order
     * @return bool
     */
    protected function addOrderItemFee(string $itemFeeName, CurrencyAmount $amount, WC_Order $order) : bool
    {
        if (0 === $amount->getAmount() || $this->orderHasItemFee($order, $itemFeeName)) {
            return false;
        }

        $convertedAmount = (new CurrencyAmountAdapter(0, ''))->convertToSource($amount);

        $item = $this->createOrderItemFee($convertedAmount, $itemFeeName);

        $order->add_item($item);
        $order->add_order_note(sprintf(
            /* translators: Placeholders: %1$s - item fee name, %2$s - item fee amount */
            __('%1$s amount of %2$s added to order by GoDaddy Payments Smart Terminal', 'mwc-core'),
            $itemFeeName,
            wc_price($convertedAmount, get_woocommerce_currency_symbol())
        ));

        return true;
    }

    /**
     * Creates a new WooCommerce order item fee with the provided amount.
     *
     * All fees created here are non-taxable and must be used to add order items like tip, cashback, etc.
     *
     * @TODO consider moving this method to a MWC Common repository method while implementing MWC-3115 {unfulvio 2021-11-02}
     *
     * @param float $amount
     * @param string $feeName
     * @return WC_Order_Item_Fee
     */
    protected function createOrderItemFee(float $amount, string $feeName) : WC_Order_Item_Fee
    {
        $item = new WC_Order_Item_Fee();
        $item->set_name($feeName);
        $item->set_amount($amount);
        $item->set_total($amount);
        $item->set_tax_status('none');

        return $item;
    }

    /**
     * Determines whether an order has an item fee with a given name.
     *
     * @TODO consider moving this method to a MWC Common repository method while implementing MWC-3115 {unfulvio 2021-11-02}
     *
     * @param WC_Order $order
     * @param string $itemFeeName
     * @return bool
     */
    protected function orderHasItemFee(WC_Order $order, string $itemFeeName) : bool
    {
        foreach ($order->get_fees() as $item) {
            if ($item instanceof WC_Order_Item_Fee && $itemFeeName === $item->get_name()) {
                return true;
            }
        }

        return false;
    }
}
