<?php

namespace GoDaddy\WordPress\MWC\Core\Features\EmailNotifications\Traits;

use Exception;
use GoDaddy\WordPress\MWC\Common\Helpers\ArrayHelper;
use GoDaddy\WordPress\MWC\Common\Register\Register;
use GoDaddy\WordPress\MWC\Common\Register\Types\RegisterFilter;
use GoDaddy\WordPress\MWC\Core\Features\EmailNotifications\Contracts\EmailTemplateContract;
use Pelago\Emogrifier;
use Pelago\Emogrifier\HtmlProcessor\HtmlPruner;
use Throwable;
use WC_Email;

/**
 * A trait for classes that generate from WooCommerce email objects.
 */
trait CanGetWooCommerceEmailOutputTrait
{
    /** @var string[] */
    protected $wooCommerceTemplateOptions = [
        'woocommerce_email_background_color'      => 'container.backgroundColor',
        'woocommerce_email_base_color'            => 'header.backgroundColor',
        'woocommerce_email_body_background_color' => 'body.backgroundColor',
        'woocommerce_email_footer_text'           => 'footer.footerText',
        'woocommerce_email_header_image'          => 'header.image.url',
        'woocommerce_email_text_color'            => 'body.text.color',
    ];

    /** @var RegisterFilter[] */
    protected $wooCommerceTemplateOptionFilters = [];

    /** @var array configuration from a {@see EmailTemplateContract} instance */
    protected $emailTemplateConfiguration = [];

    /** @var string|null CSS rules applied to WooCommerce email content */
    protected $wooCommerceEmailStyles;

    /**
     * Sets the configuration of the email template.
     *
     * @var array an configuration array likely generated with {@see EmailTemplateContract::getConfiguration()}
     */
    protected function setEmailTemplateConfiguration(array $value)
    {
        $this->emailTemplateConfiguration = $value;
    }

    /**
     * Gets the configuration of the email template.
     *
     * @return array
     */
    protected function getEmailTemplateConfiguration() : array
    {
        return $this->emailTemplateConfiguration;
    }

    /**
     * Sets the email template configuration using the information from the given template instance.
     *
     * @param EmailTemplateContract $emailTemplate optional email template instance
     */
    protected function setConfigurationFromEmailTemplate(EmailTemplateContract $emailTemplate = null)
    {
        if ($emailTemplate) {
            $this->setEmailTemplateConfiguration($emailTemplate->getConfiguration());
        }
    }

    /**
     * Adds inline style attributes for the HTML code generated by WooCommerce email hooks using WooCommerce CSS for emails.
     *
     * TODO: add tests for this method {wvega 2021-10-13}
     *
     * @see WC_Email::style_inline()
     *
     * @param string $content
     * @return string
     */
    protected function addInlineStyles(string $content) : string
    {
        $css = $this->getWooCommerceEmailStyles();

        // TODO: we should find another way to avoid using this deprecated library {nmolham 2021-10-13}
        if (class_exists('DOMDocument') && class_exists(Emogrifier::class)) {
            try {
                $emogrifier = new Emogrifier($content, $css);

                do_action('woocommerce_emogrifier', $emogrifier, $this);

                $content = $emogrifier->emogrify();

                $htmlPruner = HtmlPruner::fromHtml($content);
                $htmlPruner->removeElementsWithDisplayNone();

                $content = $htmlPruner->renderBodyContent();
            } catch (Exception $e) {
                // TODO: maybe throw a Sentry exception to report the problem? {wvega 2021-1012}
            }
        } else {
            $content = '<style type="text/css">'.$css.'</style>'.$content;
        }

        return $content;
    }

    /**
     * Gets WooCommerce CSS rules for emails.
     *
     * TODO: add tests for this method {wvega 2021-10-13}
     *
     * @return string
     */
    protected function getWooCommerceEmailStyles() : string
    {
        if (is_null($this->wooCommerceEmailStyles)) {
            $this->wooCommerceEmailStyles = $this->loadWooCommerceEmailStyles();
        }

        return $this->wooCommerceEmailStyles;
    }

    /**
     * Gets WooCommerce CSS rules for emails.
     *
     * TODO: add tests for this method {wvega 2021-10-13}
     *
     * @return string
     */
    protected function loadWooCommerceEmailStyles() : string
    {
        $styles = $this->getOutputFromCallback(function () {
            wc_get_template('emails/email-styles.php');
        });

        if (! $wooCommerceEmail = $this->getWooCommerceEmail()) {
            return $styles;
        }

        return (string) apply_filters('woocommerce_email_styles', $styles, $wooCommerceEmail);
    }

    /**
     * Gets the WooCommerce email object.
     *
     * @return WC_Email|null
     */
    abstract protected function getWooCommerceEmail();

    /**
     * Executes the given callback and returns its output.
     *
     * @param callable $callback
     * @return string
     */
    protected function getOutputFromCallback(callable $callback) : string
    {
        return (string) $this->tryOutputBufferingCallback(function () use ($callback) {
            ob_start();

            $callback();

            return (string) ob_get_clean();
        });
    }

    /**
     * Executes the given callback and catches all exceptions and errors thrown.
     *
     * It closes any output buffers that remain opened when an exception or error occurs.
     *
     * @param callable $callback a callable that returns content
     * @return string|null
     */
    protected function tryOutputBufferingCallback(callable $callback)
    {
        $outputBufferingLevel = ob_get_level();

        try {
            return $callback();
        } catch (Throwable $exception) {
            // ignore all Exceptions and Errors
        } finally {
            // make sure that all output buffers are closed
            // this finally block will be executed even if the try block returns
            while (ob_get_level() > $outputBufferingLevel) {
                ob_end_clean();
            }
        }

        return null;
    }

    /**
     * Registers filters to override the value of several WooCommerce email template options.
     */
    protected function temporarilyOverrideWooCommerceTemplateOptions()
    {
        foreach (array_keys($this->wooCommerceTemplateOptions) as $optionName) {
            /** @var RegisterFilter */
            $filter = Register::filter()
                ->setGroup("option_{$optionName}")
                ->setHandler([$this, 'overrideWooCommerceTemplateOption'])
                ->setArgumentsCount(2);

            try {
                $filter->execute();
            } catch (Exception $exception) {
                // move on if there is an error trying to register the filter
                continue;
            }

            $this->wooCommerceTemplateOptionFilters[] = $filter;
        }
    }

    /**
     * Filters the value of the given option if it is one of the WooCommerce email template options with overrides.
     *
     * @param mixed $value the current value of the option
     * @param string $option the name of the option that we are filtering
     */
    public function overrideWooCommerceTemplateOption($value, $option)
    {
        if (! $setting = ArrayHelper::get($this->wooCommerceTemplateOptions, $option)) {
            return $value;
        }

        return ArrayHelper::get($this->getEmailTemplateConfiguration(), $setting, $value);
    }

    /**
     * Deregisters the filters used to override the value WooCommerce email template options.
     */
    protected function restoreWooCommerceTemplateOptions()
    {
        foreach ($this->wooCommerceTemplateOptionFilters as $filter) {
            $filter->deregister();
        }

        $this->wooCommerceTemplateOptionFilters = [];
    }
}
