<?php

namespace GoDaddy\WordPress\MWC\Core\Features\EmailNotifications\Models;

use Exception;
use GoDaddy\WordPress\MWC\Common\Helpers\ArrayHelper;
use GoDaddy\WordPress\MWC\Common\Repositories\WooCommerce\EmailsRepository;
use GoDaddy\WordPress\MWC\Core\Features\EmailNotifications\Contracts\WooCommerceEmailNotificationContract;
use GoDaddy\WordPress\MWC\Core\Features\EmailNotifications\Traits\CanGetWooCommerceEmailOutputTrait;
use WC_Email;
use WP_Hook;

/**
 * This class will load the structured content from a WooCommerce email object.
 */
class ThirdPartyWooCommerceEmailContent extends DefaultEmailContent
{
    use CanGetWooCommerceEmailOutputTrait;

    /** @var WooCommerceEmailNotificationContract */
    protected $emailNotification;

    /** @var array */
    protected $overriddenWordPressHookHandlers = [];

    /**
     * Constructor.
     *
     * @param WooCommerceEmailNotificationContract $emailNotification
     */
    public function __construct(WooCommerceEmailNotificationContract $emailNotification)
    {
        $this->emailNotification = $emailNotification;
    }

    /**
     * {@inheritdoc}
     */
    public function getWooCommerceEmail()
    {
        return $this->emailNotification->getWooCommerceEmail();
    }

    /**
     * Gets the structured content.
     *
     * @return string
     */
    public function getStructuredContent() : string
    {
        if (! $wooCommerceEmail = $this->getWooCommerceEmail()) {
            return '';
        }

        $this->setConfigurationFromEmailTemplate($this->emailNotification->getTemplate());

        $this->temporarilyOverrideWooCommerceTemplateOptions();
        $this->temporarilyOverrideWooCommerceEmailLayout();

        $content = $this->getStructuredContentFromWooCommerceEmail($wooCommerceEmail);

        $this->restoreWooCommerceEmailLayout();
        $this->restoreWooCommerceTemplateOptions();

        return $content;
    }

    protected function getStructuredContentFromWooCommerceEmail(WC_Email $wooCommerceEmail) : string
    {
        $output = $this->tryOutputBufferingCallback(function () use ($wooCommerceEmail) {
            return $wooCommerceEmail->get_content_html();
        });

        if (is_null($output)) {
            return '';
        }

        return '<mj-column><mj-text>'.$this->addInlineStyles($output).'</mj-text></mj-column>';
    }

    public function temporarilyOverrideWooCommerceEmailLayout()
    {
        try {
            if ($mailer = EmailsRepository::mailer()) {
                $this->overrideWordPressHookHandler('woocommerce_email_header', [$mailer, 'email_header'], '__return_empty_string');
                $this->overrideWordPressHookHandler('woocommerce_email_footer', [$mailer, 'email_footer'], '__return_empty_string');
            }
        } catch (Exception $exception) {
            // do nothing
        }
    }

    protected function overrideWordPressHookHandler(string $hookName, $currentFunction, $newFunction)
    {
        if (! $hook = $this->getWordPressHook($hookName)) {
            return;
        }

        foreach ($hook->callbacks as $priority => $handlers) {
            if (! ArrayHelper::accessible($handlers)) {
                continue;
            }

            foreach ($handlers as $identifier => $handler) {
                if ($currentFunction !== ArrayHelper::get($handler, 'function')) {
                    continue;
                }

                $hookPriority = $priority;
                $hookIdentifier = $identifier;
                $hookFunction = ArrayHelper::get($handler, 'function');

                break 2;
            }
        }

        if (! isset($hookPriority, $hookIdentifier, $hookFunction)) {
            return;
        }

        $hook->callbacks[$hookPriority][$hookIdentifier]['function'] = $newFunction;

        $this->overriddenWordPressHookHandlers[$hookName][$hookPriority][$hookIdentifier] = $hookFunction;
    }

    protected function getWordPressHook(string $hookName)
    {
        // can't use ArrayHelper here because hook names can contain dots
        $hook = $GLOBALS['wp_filter'][$hookName] ?? null;

        if (! $hook instanceof WP_Hook || ! isset($hook->callbacks) || ! ArrayHelper::accessible($hook->callbacks)) {
            return null;
        }

        return $hook;
    }

    public function restoreWooCommerceEmailLayout()
    {
        $this->restoreOverridenHookHandlers();
    }

    protected function restoreOverridenHookHandlers()
    {
        foreach ($this->overriddenWordPressHookHandlers as $hookName => $callbacks) {
            if (! $hook = $this->getWordPressHook($hookName)) {
                continue;
            }

            foreach ($callbacks as $priority => $handlers) {
                foreach ($handlers as $identifier => $function) {
                    if (isset($hook->callbacks[$priority][$identifier])) {
                        $hook->callbacks[$priority][$identifier]['function'] = $function;
                    }
                }
            }
        }
    }
}
